<?php

namespace NewHeap\Component\Mautic\Connector\Application;

use NewHeap\Component\Mautic\Connector\Application\Configuration\Config;
use NewHeap\Component\Mautic\Connector\Application\Service\HttpService;
use NewHeap\Component\Mautic\Connector\Application\Service\VisualComposerService;
use NewHeap\Component\Mautic\Connector\Application\Util\Messages;

/**
 * ApplicationKernel
 *
 * Supports PHP 5.6+
 *
 * @package NewHeap\Component\Mautic\Connector
 */
final class ApplicationKernel
{
    /** @var \NewHeap\Component\Mautic\Connector\Application\ApplicationKernel */
    private static $instance;

    /** @var  Config */
    public $config;

    /** @var  VisualComposerService */
    public $visualComposerService;

    /** @var  HttpService */
    public $httpService;

    /**
     * Get the existing instance of the kernel
     *
     * @author Lars van den Bosch <lars@newheap.com>
     * @return \NewHeap\Component\Mautic\Connector\Application\ApplicationKernel
     */
    public static function getInstance()
    {
        return self::$instance;
    }

    /**
     * Kernel constructor
     *
     * @param Config $config
     * @author Lars van den Bosch <lars@newheap.com>
     * @throws \Exception
     */
    public function __construct(Config $config)
    {
        //Override config application root
        $config->applicationRoot = (__DIR__);

        $this->config = $config;

        $this->init();

        self::$instance = $this;
    }

    /**
     * Initializes the kernel, loading the core and the Application.
     *
     * @author Lars van den Bosch <lars@newheap.com>
     * @throws \Exception
     */
    private function init()
    {
        try
        {
            /**
             * Load Application
             */
            if(true !== $this->loadApplication())
            {
                throw new \Exception(Messages\ErrorMessage::APPLICATION_INITIALIZATION_FAILED);
            }

            /**
             * Execute services
             */
            $this->httpService = new HttpService();
            $this->visualComposerService = new VisualComposerService();
        }
        catch(\Exception $e)
        {
            throw new \Exception($e->getMessage());
        }
    }

    /**
     * Loads the Application, initializing the configuration and routing, preparing the kernel for ignition
     *
     * @author Lars van den Bosch <lars@newheap.com>
     * @throws \Exception
     * @return bool
     */
    private function loadApplication()
    {
        try {
            //Load application
            $aplicationFiles = $this->locateFilesFromDir($this->config->applicationRoot);

            foreach($aplicationFiles as $aplicationFile)
            {
                $filePathAndName = $aplicationFile['path'].$aplicationFile['name'];

                {
                    $fileParts = pathinfo($filePathAndName);

                    // Skip non PHP files and cached files. Cached files dont have an extension!
                    if(isset($fileParts['extension']) && $fileParts['extension'] != "php" || strpos($fileParts['dirname'], '/Application/Service/cache'))
                    {
                        continue;
                    }

                }

                require_once ($filePathAndName);
            }

            //Extra directories
            if (true === is_array($this->config->extraApplicationDirectories)) {
                foreach ($this->config->extraApplicationDirectories as $extraDir) {
                    $extraFiles = $this->locateFilesFromDir($extraDir);
                    foreach ($extraFiles as $file) {
                        $file = $file['path'] . $file['name'];
                        if (true !== file_exists($file)) {
                            return false;
                        }

                        {
                            $fileParts = pathinfo($file);

                            if($fileParts['extension'] != "php")
                            {
                                die("match");
                                continue;
                            }

                        }

                        require_once($file);
                    }
                }
            }

            return true;
        } catch (\Exception $e) {
            throw new \Exception($e->getMessage());
        }
    }

    /**
     * Set extra Application loading directories
     *
     * @author Lars van den Bosch <lars@newheap.com>
     * @param array $extraApplicationDirectories
     * @return bool
     */
    public function setExtraApplicationDirectories($extraApplicationDirectories = array())
    {
        if(true === is_array($extraApplicationDirectories))
        {
            $this->extraApplicationDirectories = $extraApplicationDirectories;
            return true;
        }

        return false;
    }

    /**
     * Locates files and file information recursively from an directory
     *
     * @author Lars van den Bosch <lars@newheap.com>
     * @param $directory
     * @param bool $includeEmptyDirs
     * @return array
     */
    private function locateFilesFromDir($directory, $includeEmptyDirs = false)
    {
        $data = array();

        //Scan directory recursive, skip dots and use leaves only.
        $it = new \RecursiveDirectoryIterator($directory, \RecursiveDirectoryIterator::SKIP_DOTS);
        $itMode = (true === $includeEmptyDirs) ? \RecursiveIteratorIterator::SELF_FIRST : \RecursiveIteratorIterator::LEAVES_ONLY;
        foreach (new \RecursiveIteratorIterator($it, $itMode) as $filename => $file)
        {
            //Parse filename to proper file name with path
            $fileName = str_replace("\\", "/", str_replace($directory, "", $filename));
            $path = str_replace("\\", "/",$directory);
            $fileDate = new \DateTime();
            $fileDate->setTimestamp($file->getMTime());

            $data[] = array(
                "name" => $fileName,
                "path" => $path,
                "size" => $file->getSize(),
                "lastModifiedDate" => $fileDate,
            );
        }

        return $data;
    }
}