<?php
namespace NewHeap\Component\Mautic\Connector\Application\Database;

use NewHeap\Component\Mautic\Connector\Application\Service\HttpService;

class FormSubmissions
{
    const FORM_SUBMISSION_TABLE = 'nh_mautic_form_submission';
    const FORM_SUBMISSION_STATE_SUCCESS = 1;

    private $wpdb;

    public function __construct()
    {
        global $wpdb;

        $this->wpdb = $wpdb;

        $this->get_not_synced_submissions();

        $this->create_database_table();
    }

    /**
     * Create database table if it doesn't exist
     *
     * @author Ferhat Ileri <ferhat@newheap.com>
     */
    public function create_database_table()
    {
        $charset_collate = $this->wpdb->get_charset_collate();
        $table_name = $this->wpdb->prefix . self::FORM_SUBMISSION_TABLE;

        // Check if the table doesn't exists. Then create the table
        if (empty($this->wpdb->get_results("SHOW TABLES LIKE '" . $table_name . "'"))) {

            $sql = "CREATE TABLE {$table_name} (
                id int(9) NOT NULL AUTO_INCREMENT,
                form_data text NOT NULL,
                post_url text NOT NULL,
                client_ip varchar(255) NOT NULL,
                visited_at datetime,
                synced boolean DEFAULT false,
                PRIMARY KEY  (id)
            ) {$charset_collate};";

            require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );
            dbDelta($sql);
        }
    }

    /**
     * Get all not synchronized submissions
     *
     * @author Ferhat Ileri <ferhat@newheap.com>
     */
    public function get_not_synced_submissions()
    {
        // TODO: get all submission where synced = 0
        $table_name = $this->wpdb->prefix . self::FORM_SUBMISSION_TABLE;
        $submissions = $this->wpdb->get_results("SELECT * FROM " . $table_name . " WHERE synced = '0';");

        // Loop through all submission
        foreach ($submissions as $submission) {
            // Resubmit submission to Mautic
            $this->resubmit_submission($submission);
        }
    }

    /**
     * Post submission to Mautic
     *
     * @param $submission
     * @author Ferhat Ileri <ferhat@newheap.com>
     */
    public function resubmit_submission($submission)
    {
        if (!wp_doing_ajax()) {
            $submission_date = \DateTime::createFromFormat('Y-m-d H:i:s', $submission->visited_at);

            // Wait before retrying
            if ((time() - 30) > $submission_date->format('U')) {
                HttpService::postRequest($submission->post_url, json_decode($submission->form_data), json_decode($submission->client_ip));

                $this->set_submission_synced($submission);
            }
        }
    }

    /**
     * Set synced to true for submission
     *
     * @param $submission
     * @author Ferhat Ileri <ferhat@newheap.com>
     */
    public function set_submission_synced($submission)
    {
        $table_name = $this->wpdb->prefix . self::FORM_SUBMISSION_TABLE;
        $data = [
            'synced' => 1
        ];
        $where = [
            'id' => $submission->id
        ];

        // Update db record
        $this->wpdb->update($table_name, $data, $where);
    }

    /**
     * Store submission in database
     *
     * @param $post_url
     * @param $form_data
     * @param $client_ip
     * @return int
     * @author Ferhat Ileri <ferhat@newheap.com>
     */
    public static function register_form_submission($post_url, $form_data, $client_ip)
    {
        global $wpdb;

        // Data to be inserted
        $data = [];

        // Get the full url with protoocol and parameters
        $data['form_data'] = json_encode($form_data);
        $data['post_url'] = $post_url;
        $data['client_ip'] = json_encode($client_ip);
        $data['visited_at'] = date('Y-m-d H:i:s');

        // Get the table name
        $table = $wpdb->prefix . self::FORM_SUBMISSION_TABLE;

        $format = ['%s','%s', '%s'];

        // Insert sanitized data
        $wpdb->insert($table, $data, $format);

        return $wpdb->insert_id;
    }

    /**
     * Update the sync state of submission
     *
     * @param $id
     * @param $state
     * @author Ferhat Ileri <ferhat@newheap.com>
     */
    public static function update_submission_sync_state($id, $state)
    {
        global $wpdb;

        $table = $wpdb->prefix . self::FORM_SUBMISSION_TABLE;

        $wpdb->update($table, [
            'synced' => $state
        ],
            [
                'id'=> $id
            ]);
    }
}
